{

   This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
}
unit FreeUtils;

// ==========================================================
//
// Delphi wrapper for FreeImage 3
//
// Design and implementation by
// - Anatoliy Pulyaevskiy (xvel84@rambler.ru)
//
// Contributors:
// - Enzo Costantini (enzocostantini@libero.it)
//
// This file is part of FreeImage 3
//
// COVERED CODE IS PROVIDED UNDER THIS LICENSE ON AN "AS IS" BASIS, WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, WITHOUT LIMITATION, WARRANTIES
// THAT THE COVERED CODE IS FREE OF DEFECTS, MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE
// OR NON-INFRINGING. THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE COVERED
// CODE IS WITH YOU. SHOULD ANY COVERED CODE PROVE DEFECTIVE IN ANY RESPECT, YOU (NOT
// THE INITIAL DEVELOPER OR ANY OTHER CONTRIBUTOR) ASSUME THE COST OF ANY NECESSARY
// SERVICING, REPAIR OR CORRECTION. THIS DISCLAIMER OF WARRANTY CONSTITUTES AN ESSENTIAL
// PART OF THIS LICENSE. NO USE OF ANY COVERED CODE IS AUTHORIZED HEREUNDER EXCEPT UNDER
// THIS DISCLAIMER.
//
// Use at your own risk!
//
// ==========================================================

interface

uses
    SysUtils, Classes, FreeImage;

function FIU_GetFIFType(filename: string): FREE_IMAGE_FORMAT;

// returns FIF (plugin) description string
function FIU_GetFIFDescription(fif: FREE_IMAGE_FORMAT): string;

procedure FIU_GetAllDescriptions(var Descriptions: TStringList);

// returns file extentions for FIF (e.g. '*.tif;*.tiff)
function FIU_GetFIFExtList(fif: FREE_IMAGE_FORMAT): string;

// returns file extentions for all plugins
function FIU_GetFullExtList: string;

// returns "Description + | + ExtList" for specified FIF
function FIU_GetFIFFilter(fif: FREE_IMAGE_FORMAT): string;

// All supported formats + Full filter list for FIFs + All files
function FIU_GetAllFilters: string;

//߰
function FIU_GetAllFilters2: string;
function FIU_GetAllFilters3: string;
function FIU_GetSelectExt(idx:integer): string;
function FIU_GetSelectidx(ext:string): integer;
function FIU_GetFIFType2(ext:string): FREE_IMAGE_FORMAT;

implementation

uses StrUtils;

function FIU_GetFIFType(filename: string): FREE_IMAGE_FORMAT;
begin
  Result:=FreeImage_GetFileType(PAnsiChar(filename),0);
end;

function FIU_GetFIFDescription(fif: FREE_IMAGE_FORMAT): string;
begin
  Result := FreeImage_GetFIFDescription(fif)
end;

procedure FIU_GetAllDescriptions(var Descriptions: TStringList);
var
  fif: FREE_IMAGE_FORMAT;
begin
  Descriptions.Clear;
  for fif := FIF_BMP to FIF_GIF do
    Descriptions.Add(FreeImage_GetFIFDescription(fif) + ' (' + FIu_GetFIFExtList(fif) + ')')
end;

function FIU_GetFIFExtList(fif: FREE_IMAGE_FORMAT): string;
var
  ExtList: string;
  I: Smallint;
  C: Char;
begin
  Result := '*.';
  ExtList := FreeImage_GetFIFExtensionList(fif);
  for I := 1 to Length(ExtList) do
  begin
    C := ExtList[i];
    if C <> ',' then
      Result := Result + C
    else
      Result := Result + ';*.';
  end
end;

function FIU_GetFullExtList: string;
var
  fif: FREE_IMAGE_FORMAT;
begin
  Result := FIU_GetFIFExtList(FIF_BMP);
  for fif := FIF_ICO to FIF_GIF do
    Result := Result + ';' + FIU_GetFIFExtList(fif)
end;

function FIU_GetFIFFilter(fif: FREE_IMAGE_FORMAT): string;
var
  Text, ExtList: string;
begin
  Result := '';
  if fif <> FIF_UNKNOWN then
  begin
    Text := Trim(FreeImage_GetFIFDescription(fif));
    ExtList := FIU_GetFIFExtList(fif);
    Result := Text + '(' + ExtList + ')' + '|' + ExtList
  end
end;

function FIU_GetAllFilters: string;
var
  fif: FREE_IMAGE_FORMAT;
begin
  Result := 'All supported formats|' + FIU_GetFullExtList;
  for fif := FIF_BMP to FIF_GIF do
  begin
     Result := Result + '|' + FIU_GetFIFFilter(fif)
  end;
end;
//߰
function FIU_GetAllFilters2: string;
var
  fif: FREE_IMAGE_FORMAT;
begin
  for fif := FIF_BMP to FIF_GIF do
  begin
    if FreeImage_FIFSupportsWriting(fif) then
      Result := Result + FIU_GetFIFFilter(fif) + '|';
  end;
end;

function FIU_GetAllFilters3: string;
var
  fif: FREE_IMAGE_FORMAT;
begin
  for fif := FIF_BMP to FIF_GIF do
  begin
    if FreeImage_FIFSupportsWriting(fif) then
      Result := Result + FIU_GetFIFFilter(fif) + '^';
  end;
end;

function FIU_GetSelectidx(ext:string): integer;
var
  fif: FREE_IMAGE_FORMAT;
  s:string;
  k:integer;
begin
  k:=0;
  if (length(ext)>0) and (ext[1]='.') then
    delete(ext,1,1);
  for fif := FIF_BMP to FIF_GIF do
    if FreeImage_FIFSupportsWriting(fif) then begin
       inc(k);
       s:=FreeImage_GetFIFExtensionList(fif);
       if pos(ext,s)>0 then begin
         result:=k;
         break;
       end;
    end;
end;

function FIU_GetSelectExt(idx:integer): string;
var
  fif: FREE_IMAGE_FORMAT;
  k,p1:integer;
  s:string;
begin
  k:=0;
  for fif := FIF_BMP to FIF_GIF do
    if FreeImage_FIFSupportsWriting(fif) then begin
      inc(k);
      if k=idx then begin
        s:=FIU_GetFIFExtList(fif);
        p1:=pos(';',s);
        if p1>0 then s:=copy(s,1,p1-1);
        s:=sysutils.StringReplace(s,'*','',[rfReplaceAll]);
        result:=s;
        break;
      end;
    end;
end;

function FIU_GetFIFType2(ext:string): FREE_IMAGE_FORMAT;
var
  fif: FREE_IMAGE_FORMAT;
  s:string;
begin
  if (length(ext)>0) and (ext[1]='.') then
    delete(ext,1,1);

  for fif := FIF_BMP to FIF_GIF do begin
    s:=FreeImage_GetFIFExtensionList(fif);
    if pos(ext,s)>0 then begin
      result:=fif;
      break;
    end;
  end;
end;

end.
